// --- AI SECURITY: Encryption/Decryption for API Keys ---
// Simple XOR encryption - not military-grade, but prevents casual viewing
const AI_SECURITY_KEY = 'webvidra-ai-secure-2025';

function simpleEncrypt(text, key) {
    let result = '';
    for (let i = 0; i < text.length; i++) {
        result += String.fromCharCode(text.charCodeAt(i) ^ key.charCodeAt(i % key.length));
    }
    return btoa(result); // Base64 encode
}

function simpleDecrypt(encoded, key) {
    try {
        let text = atob(encoded); // Base64 decode
        let result = '';
        for (let i = 0; i < text.length; i++) {
            result += String.fromCharCode(text.charCodeAt(i) ^ key.charCodeAt(i % key.length));
        }
        return result;
    } catch (e) {
        console.warn('Decryption failed:', e);
        return '';
    }
}

function saveAIKeyToStorage(apiKey) {
    try {
        if (!apiKey) {
            localStorage.removeItem('ai-api-key-encrypted');
            return;
        }
        const encrypted = simpleEncrypt(apiKey, AI_SECURITY_KEY);
        localStorage.setItem('ai-api-key-encrypted', encrypted);
        console.log('AI API key saved securely to localStorage');
    } catch (e) {
        console.warn('Could not save AI key:', e);
    }
}

function loadAIKeyFromStorage() {
    try {
        const encrypted = localStorage.getItem('ai-api-key-encrypted');
        if (!encrypted) return '';
        return simpleDecrypt(encrypted, AI_SECURITY_KEY);
    } catch (e) {
        console.warn('Could not load AI key:', e);
        return '';
    }
}

function clearAIKeyFromStorage() {
    try {
        localStorage.removeItem('ai-api-key-encrypted');
        console.log('AI API key cleared from localStorage');
    } catch (e) {
        console.warn('Could not clear AI key:', e);
    }
}

// --- AI ASSISTANT CHAT WIDGET ---

// This function is called by Python when AI finishes processing
eel.expose(on_ai_finished);
function on_ai_finished(result) {
    console.log("AI finished:", result);
    
    // Remove loading message
    let messagesContainer = document.getElementById('ai-messages');
    let lastMsg = messagesContainer.lastChild;
    if (lastMsg && lastMsg.classList.contains('ai-loading')) {
        lastMsg.remove();
    }
    
    // Display result
    if (result.success) {
        addAIMessage(`✓ ${result.message} (${result.blocks_modified || 0} changes)`, 'ai-response');
        
        // IMPORTANT: Reload project data from Python to sync with new blocks
        // This ensures appData is updated with the blocks that Python saved
        loadProject().then(() => {
            console.log("Project reloaded after AI");
            renderCanvas(); // Now render with updated data
            
            // If global settings modal is open, refresh it too
            if (document.getElementById('settings-overlay') && document.getElementById('settings-overlay').style.display !== 'none') {
                openGlobalSettings();
            }
        });
    } else {
        addAIMessage(`✗ Error: ${result.message}`, 'ai-error');
    }
    
    // Re-enable send button and keep AI window in focus
    document.getElementById('ai-send-btn').disabled = false;
    
    // Ensure AI chat window is visible and scroll to bottom
    let chatWindow = document.getElementById('ai-chat-window');
    if (chatWindow.style.display === 'none') {
        chatWindow.style.display = 'flex';
    }
    
    // Scroll chat messages to bottom to show latest response
    setTimeout(() => {
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
        document.getElementById('ai-input').focus();
    }, 100);
}


function toggleAIChat() {
    let chatWindow = document.getElementById('ai-chat-window');
    if (chatWindow.style.display === 'none') {
        chatWindow.style.display = 'flex';
        document.getElementById('ai-input').focus();
    } else {
        chatWindow.style.display = 'none';
    }
}

async function repairProjectDataManual() {
    let repairBtn = document.getElementById('ai-repair-btn');
    if (!repairBtn) return;
    
    repairBtn.disabled = true;
    addAIMessage('🔧 Repairing project data structure...', 'ai-loading');
    
    try {
        let result = await eel.repair_project_data_manual()();
        
        // Remove loading message
        let messagesContainer = document.getElementById('ai-messages');
        let lastMsg = messagesContainer.lastChild;
        if (lastMsg && lastMsg.classList.contains('ai-loading')) {
            lastMsg.remove();
        }
        
        if (result.success) {
            addAIMessage(`✓ ${result.message}`, 'ai-response');
            // Reload project to show any changes
            await loadProject();
            renderCanvas();
        } else {
            addAIMessage(`✗ Repair failed: ${result.message}`, 'ai-error');
        }
    } catch (err) {
        let messagesContainer = document.getElementById('ai-messages');
        let lastMsg = messagesContainer.lastChild;
        if (lastMsg && lastMsg.classList.contains('ai-loading')) {
            lastMsg.remove();
        }
        console.error('Repair error:', err);
        addAIMessage(`✗ Repair error: ${err}`, 'ai-error');
    }
    
    repairBtn.disabled = false;
}

document.addEventListener('DOMContentLoaded', () => {
    let toggleBtn = document.getElementById('ai-toggle-btn');
    if (toggleBtn) {
        toggleBtn.addEventListener('click', toggleAIChat);
    }
    
    // Restore AI chat history from localStorage
    restoreAIChatHistory();
});

function addAIMessage(text, type = 'ai-response') {
    let messagesContainer = document.getElementById('ai-messages');
    if (!messagesContainer) return;
    
    let messageDiv = document.createElement('div');
    messageDiv.className = `ai-message ${type}`;
    messageDiv.textContent = text;
    
    messagesContainer.appendChild(messageDiv);
    // Auto-scroll to bottom
    messagesContainer.scrollTop = messagesContainer.scrollHeight;
    
    // Save message to localStorage for persistence
    saveAIChatMessage(text, type);
}

function saveAIChatMessage(text, type) {
    try {
        let history = JSON.parse(localStorage.getItem('ai-chat-history') || '[]');
        history.push({
            text: text,
            type: type,
            timestamp: new Date().toISOString()
        });
        localStorage.setItem('ai-chat-history', JSON.stringify(history));
    } catch (e) {
        console.warn('Could not save AI chat message:', e);
    }
}

function restoreAIChatHistory() {
    try {
        let history = JSON.parse(localStorage.getItem('ai-chat-history') || '[]');
        let messagesContainer = document.getElementById('ai-messages');
        
        if (!messagesContainer || history.length === 0) return;
        
        // Clear default system message (only keep if no history)
        if (history.length > 0) {
            messagesContainer.innerHTML = '';
        }
        
        // Restore all messages
        history.forEach(msg => {
            let messageDiv = document.createElement('div');
            messageDiv.className = `ai-message ${msg.type}`;
            messageDiv.textContent = msg.text;
            messagesContainer.appendChild(messageDiv);
        });
        
        // Scroll to bottom
        messagesContainer.scrollTop = messagesContainer.scrollHeight;
    } catch (e) {
        console.warn('Could not restore AI chat history:', e);
    }
}

function clearAIChatHistory() {
    try {
        localStorage.removeItem('ai-chat-history');
        let messagesContainer = document.getElementById('ai-messages');
        if (messagesContainer) {
            messagesContainer.innerHTML = '<div class="ai-message ai-system"><p>Hello! I can help you edit your project. Describe what you\'d like to change, and I\'ll modify the blocks for you.</p></div>';
        }
    } catch (e) {
        console.warn('Could not clear AI chat history:', e);
    }
}

async function sendAIMessage() {
    let input = document.getElementById('ai-input');
    let sendBtn = document.getElementById('ai-send-btn');
    
    if (!input.value.trim()) return;
    
    let userMessage = input.value.trim();
    input.value = '';
    
    // Display user message
    addAIMessage(userMessage, 'ai-user');
    
    // Disable send button and show loading
    sendBtn.disabled = true;
    addAIMessage('🤔 Thinking... (the local version might be slow...)', 'ai-loading');
    
    try {
        // Load AI configuration from localStorage
        let aiProvider = localStorage.getItem('ai-provider') || 'ollama';
        let aiModel = localStorage.getItem('ai-model') || 'phi3';
        let apiKey = loadAIKeyFromStorage() || '';
        
        console.log('Sending AI message with provider:', aiProvider, 'model:', aiModel, 'api_key length:', apiKey.length);
        
        // START the AI request (don't wait for response!)
        // The Python function starts a background thread and returns immediately
        let startResponse = await eel.ask_ai_to_edit(userMessage, aiProvider, aiModel, apiKey)();
        console.log("AI Request sent:", startResponse);
        
        // Don't do anything here - the work happens in background.
        // When Python finishes, it will call on_ai_finished() callback above.
        
    } catch (err) {
        // Remove loading message on error
        let messagesContainer = document.getElementById('ai-messages');
        let lastMsg = messagesContainer.lastChild;
        if (lastMsg && lastMsg.classList.contains('ai-loading')) {
            lastMsg.remove();
        }
        
        console.error('AI start error:', err);
        let errorMsg = 'Connection error';
        if (err) {
            if (err.message) errorMsg = err.message;
            else if (typeof err === 'string') errorMsg = err;
            else errorMsg = String(err);
        }
        addAIMessage(`✗ ${errorMsg}`, 'ai-error');
        sendBtn.disabled = false;
    }
}

// --- STABILITÁS ÉS ALVÓ MÓD KEZELÉS ---

let heartbeatTimeout = null;
let missedPongs = 0;
let lastSuccessfulPing = Date.now();

function startHeartbeat() {
    // Ha már futna valami, állítsuk le
    if (heartbeatTimeout) clearTimeout(heartbeatTimeout);
    missedPongs = 0;
    lastSuccessfulPing = Date.now();
    console.log("Heartbeat started (Safe Mode)...");

    const performCheck = async () => {
        // Ha az ablak nem aktív vagy bezárul, ne pingeljen
        if (document.hidden || isWindowBeingClosed) {
            missedPongs = 0;
            heartbeatTimeout = setTimeout(performCheck, 10000);
            return;
        }

        try {
            // 1. Megnövelt türelmi idő: 8 másodperc a válaszra
            const timeoutPromise = new Promise((_, reject) => 
                setTimeout(() => reject(new Error('Timeout')), 8000)
            );
            
            // Ping küldése
            const response = await Promise.race([
                eel.ping()(),
                timeoutPromise
            ]);

            if (response === "pong") {
                // Sikeres ping után, ha hosszú ideje nem volt válasz = alvó módból ébredés
                // De CSAK akkor újratöltünk, ha sok pong hiányzik
                if (missedPongs >= 5) {
                    console.log("Ébredés alvó módból! Oldal újratöltése...");
                    // Ne kellene, hogy ilyenkor újratöltsünk, mivel visibilitychange már kezeli
                    // De ha a visibilitychange nem dolgozik jól, ez a fallback
                    await new Promise(resolve => setTimeout(resolve, 500));
                    location.reload();
                    return;
                }
                
                missedPongs = 0;
                lastSuccessfulPing = Date.now();
            }

        } catch (e) {
            missedPongs++;
            console.warn(`Kapcsolat hiba (${missedPongs}/10):`, e);
            
            // Ha 3 percig nincs válasz, erőltetett újratöltés
            if (Date.now() - lastSuccessfulPing > 180000) {
                console.error("3 percig nincs kapcsolat. Erőltetett újratöltés...");
                location.reload();
                return;
            }
            
            // Csak 10 egymás utáni hiba után frissítsen (kb. 1.5 - 2 perc szakadás)
            if (missedPongs >= 10) {
                console.error("A kapcsolat tényleg megszakadt. Újratöltés...");
                location.reload();
                return; // Ne indítsuk újra a ciklust, mert reload lesz
            }
        }

        // KULCSFONTOSSÁGÚ VÁLTOZÁS:
        // Nem setInterval-t használunk, hanem a válasz beérkezése UTÁN
        // várjuk meg a következő kört. Így sosem torlódnak fel a kérések.
        // 10 másodpercenként ellenőrzünk 5 helyett.
        heartbeatTimeout = setTimeout(performCheck, 10000);
    };

    // Első ellenőrzés indítása
    performCheck();
}

function stopHeartbeat() {
    if (heartbeatTimeout) {
        clearTimeout(heartbeatTimeout);
        heartbeatTimeout = null;
        console.log("Heartbeat paused for dialog.");
    }
}
// --- SMART STARTUP ---
async function initApp() {
    try {
        // RESET: Ha alvó módból jöttünk, most reseteljük a flageket
        // (az oldal újratöltődött, új session)
        if (isReloadingDueToWakeup) {
            console.log("Alvó mód után újratöltés befejezve - flagek resetelve");
            isReloadingDueToWakeup = false;
            isWindowBeingClosed = false;
        }
        
        // 1. Először megnézzük, él-e a kapcsolat és van-e projekt
        // (Itt még NEM indítjuk a heartbeat-et, hogy ne zavarjon be)
        const isLoaded = await eel.check_if_project_loaded()();

        if (isLoaded) {
            console.log("Meglévő munkamenet észlelve. Visszatöltés...");
            
            // Eltüntetjük a nyitó ablakot
            document.getElementById('startup-overlay').style.display = 'none';
            document.querySelector('.app-container').style.display = 'flex';
            
            // Betöltjük az adatokat
            await loadProject();
            showToast("Munkamenet helyreállítva", 3000);
            
            // Helyreállítjuk az alvó mód után nyitva maradt dialógusokat
            restoreDialogsAfterWakeup();
        } else {
            console.log("Nincs betöltött projekt, várakozás a felhasználóra.");
        }

        // 2. SIKER! Most már biztonságos elindítani a figyelést
        startHeartbeat();

    } catch (e) {
        console.log("Hiba az inicializálásnál (lehet, hogy a Python még tölt):", e);
        // Ha hiba volt, próbáljuk újra 1 mp múlva (nem reload, csak retry)
        setTimeout(initApp, 1000);
    }
}

// Az oldal betöltésekor indítjuk az okos inicializálást
window.addEventListener('load', initApp);

// --- ABLAK BEZÁRÁS JELE ---
// Szól a Pythonnak, ha az ablak/böngésző tényleg bezárul
// DE csak akkor, ha nem a sleep mode után újratöltés miatt történik
let isReloadingDueToWakeup = false;
let isWindowBeingClosed = false;
let isPageRefresh = false; // Track if this is a page refresh (F5) vs actual close

// Detect if user is refreshing the page (Ctrl+R, Cmd+R, F5) or navigating away
window.addEventListener('keydown', (e) => {
    if ((e.key === 'F5') || (e.ctrlKey && e.key === 'r') || (e.metaKey && e.key === 'r')) {
        isPageRefresh = true;
        console.log("Page refresh detected (F5 or Ctrl+R)");
    }
});

window.addEventListener('beforeunload', async () => {
    // If it's a page refresh (F5), keep the Python running - it's not a real close
    if (isPageRefresh) {
        console.log("Page refresh detected - keeping Python alive");
        // Clean up heartbeat gracefully but DON'T signal close
        stopHeartbeat();
        // Prevent reload loop detection
        isWindowBeingClosed = false;
        return;
    }
    
    // If awakening from sleep mode, don't send close signal
    if (isReloadingDueToWakeup) {
        console.log("Reload after sleep - no close signal");
        return;
    }
    
    // Mark that window is closing (real close, not refresh)
    isWindowBeingClosed = true;
    
    try {
        // Signal Python that user is intentionally closing the app
        await eel.signal_window_close()();
        console.log("Bezárási jel elküldve a Pythonnak");
    } catch (e) {
        // Ha nincs kapcsolat már, az ok - az ablak úgyis bezárul
        console.log("Nem sikerült a bezárási jel küldése (valószínűleg már nincs kapcsolat)");
    }
});

// --- ALVÓ MÓD ÉBREDÉS DETEKTÁLÁSA ---
// Ha a böngésző háttérből előjön UTÁN hosszú ideig = alvó mód
// Rövid fokuszváltás (Alt+Tab) = ignore
// FONTOS: Beállítunk egy timer-t is a hosszú elrejtésre (visibilitychange késedelmei miatt)
let hiddenSince = null;
let hiddenCheckTimer = null;
let isWindowFocusChange = false; // Track focus changes vs sleep

// Detectálunk aktív fokuszváltásokat is (focus/blur)
window.addEventListener('focus', () => {
    isWindowFocusChange = true;
});

window.addEventListener('blur', () => {
    isWindowFocusChange = false;
});

document.addEventListener('visibilitychange', async () => {
    if (document.hidden) {
        // Háttérbe megy - jegyezzük meg az időpontot
        hiddenSince = Date.now();
        
        // Elindítunk egy timer-t: ha 6 másodperc múlva még mindig rejtett, akkor alvó mód
        // (Ez kezeli a lassú visibilitychange eseményeket)
        hiddenCheckTimer = setTimeout(() => {
            if (document.hidden && hiddenSince && (Date.now() - hiddenSince > 5500)) {
                console.log("Hosszú elrejtés detektálva (timer alapján) - valószínűleg alvó mód");
                hiddenCheckTimer = null;
            }
        }, 6000);
        
        console.log("Oldal háttérbe ment");
    } else if (hiddenSince && !isWindowBeingClosed) {
        // Előjön a háttérből
        const hiddenDuration = Date.now() - hiddenSince;
        hiddenSince = null;
        
        if (hiddenCheckTimer) {
            clearTimeout(hiddenCheckTimer);
            hiddenCheckTimer = null;
        }
        
        // Ha rövid ideig volt rejtett (< 6 másodperc), csak ablakváltás volt = NE újratöltsen
        // Ha hosszabb volt (alvó mód), AKKOR újratöltsen és mentse a dialógusokat
        if (hiddenDuration > 6000) {
            console.log(`Oldal ${(hiddenDuration/1000).toFixed(1)}s után felébredt! Alvó módból ébredés - újratöltés...`);
            
            // Mentjük a nyitott dialógusok állapotát CSAK alvó módnál
            const settingsOverlay = document.getElementById('settings-overlay');
            
            const openDialogs = {
                hasSettings: settingsOverlay && settingsOverlay.style.display !== 'none'
            };
            
            // Mentés localStorage-ba
            localStorage.setItem('openDialogsAfterWakeup', JSON.stringify(openDialogs));
            console.log("Dialógusok mentve alvó mód után:", openDialogs);
            
            // FONTOS: Jelezzük, hogy alvó mód miatt újratöltünk
            // így a beforeunload esemény nem küld close signalt
            isReloadingDueToWakeup = true;
            
            await new Promise(resolve => setTimeout(resolve, 500));
            location.reload();
        } else {
            console.log(`Rövid fokuszváltás (${hiddenDuration}ms) - nincs újratöltés, dialógusok megmaradnak`);
        }
    }
});

// --- DIALÓGUSOK HELYREÁLLÍTÁSA ALVÓ MÓD UTÁN ---
function restoreDialogsAfterWakeup() {
    const savedDialogs = localStorage.getItem('openDialogsAfterWakeup');
    if (!savedDialogs) {
        console.log("Nincsenek mentett dialógusok");
        return;
    }
    
    try {
        const dialogs = JSON.parse(savedDialogs);
        console.log("Dialógusok helyreállítása:", dialogs);
        
        if (dialogs.hasSettings) {
            console.log("Settings panel helyreállítása...");
            const settingsOverlay = document.getElementById('settings-overlay');
            if (settingsOverlay) {
                settingsOverlay.style.display = 'flex';
                console.log("Settings panel megjelenítve");
            }
        }
        
        // Törlés után
        localStorage.removeItem('openDialogsAfterWakeup');
    } catch (e) {
        console.warn("Hiba a dialógusok helyreállításánál:", e);
    }
}

// --- INICIALIZÁLÁS F5 UTÁN ---
function restoreStateAfterPageRefresh() {
    // Helyreállítjuk az undo/redo történetet
    const historyRestored = restoreHistoryFromStorage();
    console.log("Undo/redo history restored:", historyRestored);
}


let appData = null;
let selectedBlockId = null;
let sortableInstance = null;

// --- UNDO/REDO SYSTEM ---
let history = [];
let historyIndex = -1;
const MAX_HISTORY = 50;
const HISTORY_STORAGE_KEY = 'undo-redo-history';

function saveToHistory() {
    // Remove any redo history if we're not at the end
    if (historyIndex < history.length - 1) {
        history = history.slice(0, historyIndex + 1);
    }
    // Add current state
    let state = JSON.parse(JSON.stringify(appData));
    history.push(state);
    historyIndex = history.length - 1;
    // Keep history size under control
    if (history.length > MAX_HISTORY) {
        history.shift();
        historyIndex--;
    }
    console.log("History saved. Index:", historyIndex, "Total states:", history.length);
    console.log("Saved state:", state);
    
    // Persist history to localStorage
    persistHistoryToStorage();
}

function persistHistoryToStorage() {
    try {
        const historyData = {
            history: history,
            historyIndex: historyIndex,
            timestamp: new Date().toISOString()
        };
        localStorage.setItem(HISTORY_STORAGE_KEY, JSON.stringify(historyData));
    } catch (e) {
        console.warn('Could not save undo/redo history to localStorage:', e);
    }
}

function restoreHistoryFromStorage() {
    try {
        const stored = localStorage.getItem(HISTORY_STORAGE_KEY);
        if (stored) {
            const historyData = JSON.parse(stored);
            history = historyData.history || [];
            historyIndex = historyData.historyIndex || -1;
            console.log("Undo/redo history restored from localStorage");
            return true;
        }
    } catch (e) {
        console.warn('Could not restore undo/redo history from localStorage:', e);
    }
    return false;
}

function clearHistoryStorage() {
    try {
        localStorage.removeItem(HISTORY_STORAGE_KEY);
    } catch (e) {
        console.warn('Could not clear undo/redo history storage:', e);
    }
}

function undo() {
    console.log("Undo clicked. Current index:", historyIndex, "Total states:", history.length);
    if (historyIndex > 0) {
        historyIndex--;
        appData = JSON.parse(JSON.stringify(history[historyIndex]));
        selectedBlockId = null;
        console.log("Undone to index:", historyIndex);
        console.log("Restored appData:", appData);
        // Update page selector in case pages were added/deleted
        updatePageSelector();
        document.getElementById('properties-panel').innerHTML = '';
        saveProject(false);
        renderCanvas();
        persistHistoryToStorage();
    } else {
        console.log("Cannot undo - at beginning of history");
    }
}

function redo() {
    console.log("Redo clicked. Current index:", historyIndex, "Total states:", history.length);
    if (historyIndex < history.length - 1) {
        historyIndex++;
        appData = JSON.parse(JSON.stringify(history[historyIndex]));
        selectedBlockId = null;
        console.log("Redone to index:", historyIndex);
        console.log("Restored appData:", appData);
        // Update page selector in case pages were added/deleted
        updatePageSelector();
        document.getElementById('properties-panel').innerHTML = '';
        saveProject(false);
        renderCanvas();
        persistHistoryToStorage();
    } else {
        console.log("Cannot redo - at end of history");
    }
}

// --- INITIALIZATION ---
async function selectFolder() {
    // 1. LEÁLLÍTJUK a figyelést, mert a felugró ablak blokkolni fogja a Pythont
    stopHeartbeat();
    
    try {
        let success = await eel.select_working_directory()();
        if (success) {
            loadProject();
            // Ha sikeres, a loadProject végén vagy az initApp újraindítja majd,
            // de a biztonság kedvéért itt is újraindíthatjuk, ha kell.
        }
    } catch (e) {
        alert("Error: " + e);
    } finally {
        document.getElementById('startup-overlay').style.display = 'none';
        document.querySelector('.app-container').style.display = 'flex';
        
        // 2. VISSZAKAPCSOLJUK a figyelést, ha végeztünk
        startHeartbeat();
    }
}

async function loadProject() {
    appData = await eel.get_project_data()();
    
    // IMPORTANT: Remove ai_settings from appData if it exists (should only be in localStorage)
    if (appData.ai_settings) {
        console.log("[Security] Removing ai_settings from appData (should be in localStorage only)");
        delete appData.ai_settings;
    }
    
    let currentPage = await eel.get_current_page()();
    updatePageSelector();
    if (!appData.pages[currentPage]) {
        currentPage = "index";
        await eel.set_current_page("index")();
    }
    document.getElementById('page-select').value = currentPage;
    
    // Try to restore undo/redo history from localStorage
    const historyRestored = restoreHistoryFromStorage();
    
    // If no history in storage, initialize fresh
    if (!historyRestored) {
        history = [];
        historyIndex = -1;
        saveToHistory();
    }
    
    renderCanvas();
}

// --- GLOBAL SETTINGS LOGIC ---
function openGlobalSettings() {
    document.getElementById('gs-title').value = appData.site_title || "";
    document.getElementById('gs-site-url').value = appData.site_url || appData.site_url || "";
    document.getElementById('gs-subtitle').value = appData.site_subtitle || "";
    document.getElementById('gs-description').value = appData.site_description || "";
    document.getElementById('gs-footer').value = appData.footer_text || "";
    document.getElementById('gs-theme').value = appData.theme || "Dark";
    document.getElementById('gs-font').value = appData.font || "Segoe UI";
    
    // Load AI Configuration from localStorage (provider & model stored in browser, NOT in project file)
    let storedProvider = localStorage.getItem('ai-provider') || 'ollama';
    let storedModel = localStorage.getItem('ai-model') || 'phi3';
    let storedApiKey = loadAIKeyFromStorage() || '';
    
    document.getElementById('ai-provider').value = storedProvider;
    document.getElementById('ai-model').value = storedModel;
    document.getElementById('ai-api-key').value = storedApiKey;
    
    // Trigger UI update to show/hide fields and set placeholders
    updateAIProviderUI();
    
    renderGlobalProjectsList();
    document.getElementById('settings-overlay').style.display = 'flex';
}

function renderGlobalProjectsList() {
    let list = document.getElementById('gs-projects-list');
    list.innerHTML = '';
    (appData.projects || []).forEach((proj, idx) => {
        let d = document.createElement('div');
        d.innerHTML = `<div style="display:flex; justify-content:space-between; margin:5px 0; border-bottom:1px solid #444;">
            <span>${proj}</span> <button class="btn-danger" style="padding:0 5px;" onclick="removeGlobalProject(${idx})">X</button>
        </div>`;
        list.appendChild(d);
    });
}
window.addGlobalProject = function() {
    let val = document.getElementById('gs-new-project').value;
    if(val) {
        if(!appData.projects) appData.projects = [];
        appData.projects.push(val);
        document.getElementById('gs-new-project').value = '';
        renderGlobalProjectsList();
        saveToHistory();
    }
};
window.removeGlobalProject = function(idx) {
    appData.projects.splice(idx, 1);
    renderGlobalProjectsList();
    saveToHistory();
};
window.saveGlobalSettings = function() {
    appData.site_title = document.getElementById('gs-title').value;
    appData.site_url = document.getElementById('gs-site-url').value;
    appData.site_subtitle = document.getElementById('gs-subtitle').value;
    appData.site_description = document.getElementById('gs-description').value;
    appData.footer_text = document.getElementById('gs-footer').value;
    appData.theme = document.getElementById('gs-theme').value;
    appData.font = document.getElementById('gs-font').value;
    
    // Save AI Configuration ONLY to encrypted localStorage (NOT to project file!)
    let aiProvider = document.getElementById('ai-provider').value || 'ollama';
    let aiModel = document.getElementById('ai-model').value || 'phi3';
    let apiKey = document.getElementById('ai-api-key').value || '';
    
    // Save to localStorage (model and provider unencrypted, api_key encrypted)
    localStorage.setItem('ai-provider', aiProvider);
    localStorage.setItem('ai-model', aiModel);
    saveAIKeyToStorage(apiKey);
    
    // IMPORTANT: Remove ai_settings from project file if it exists
    if (appData.ai_settings) {
        delete appData.ai_settings;
    }
    
    document.getElementById('settings-overlay').style.display = 'none';
    saveToHistory();
    saveProject(false);
};

// AI Provider UI Handler
window.updateAIProviderUI = function() {
    let provider = document.getElementById('ai-provider').value;
    let keyWrapper = document.getElementById('ai-key-wrapper');
    let keyHint = document.getElementById('ai-key-hint');
    let modelInput = document.getElementById('ai-model');
    
    // Show/hide API key input based on provider
    if (provider === 'ollama') {
        keyWrapper.style.display = 'none';
        // Set default model for Ollama
        if (!modelInput.value || modelInput.value === 'llama3-70b-8192' || modelInput.value === 'gemini-1.5-flash') {
            modelInput.value = 'phi3';
        }
        modelInput.placeholder = 'e.g., phi3, llama3, mistral';
    } else {
        keyWrapper.style.display = 'block';
        
        // Set default models and hints based on cloud provider
        if (provider === 'groq') {
            keyHint.textContent = 'Get your API key from https://console.groq.com';
            document.getElementById('ai-api-key').placeholder = 'gsk_...';
            
            // Set default model for Groq if not already set to a Groq model
            if (!modelInput.value || modelInput.value === 'phi3' || modelInput.value === 'gemini-1.5-flash') {
                modelInput.value = 'llama3-70b-8192';
            }
            modelInput.placeholder = 'e.g., llama3-70b-8192, mixtral-8x7b-32768';
        } else if (provider === 'gemini') {
            keyHint.textContent = 'Get your API key from https://aistudio.google.com';
            document.getElementById('ai-api-key').placeholder = 'AIza...';
            
            // Set default model for Gemini if not already set to a Gemini model
            if (!modelInput.value || modelInput.value === 'phi3' || modelInput.value === 'llama3-70b-8192') {
                modelInput.value = 'gemini-1.5-flash';
            }
            modelInput.placeholder = 'e.g., gemini-pro, gemini-1.5-flash';
        }
    }
};

// --- RENDERING (IFRAME MAGIC) ---
async function renderCanvas() {
    let currentPage = document.getElementById('page-select').value;
    let blockIds = appData.pages[currentPage] || [];
    
    // Get Iframe Document
    let iframe = document.getElementById('preview-frame');
    let doc = iframe.contentDocument || iframe.contentWindow.document;
    
    // Reset Iframe Content
    doc.open();
    doc.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <link rel="stylesheet" href="preview.css?v=${new Date().getTime()}">
            <style>
                body { min-height: 100vh; padding-bottom: 100px; }
                .editor-block { position: relative; border: 2px dashed rgba(255,255,255,0.1); margin-bottom: 10px; min-height: 50px; }
                .editor-block:hover { border-color: #007acc; cursor: pointer; }
                .editor-block.selected { border: 2px solid #007acc; box-shadow: 0 0 15px rgba(0,122,204,0.5); }
                /* Only disable anchor navigation inside blocks (keep other interactions enabled) */
                .editor-block a, .editor-block a * { pointer-events: none; }
                .interactive-gallery-container { pointer-events: auto !important; }
                .interactive-gallery-container * { pointer-events: auto !important; }
                .gallery-edit-controls { pointer-events: auto !important; }
                .gallery-edit-controls * { pointer-events: auto !important; }
                .gallery-preview-list { pointer-events: auto !important; }
                .gallery-preview-list * { pointer-events: auto !important; }
                .gallery-preview-card { pointer-events: auto !important; }
            </style>
        </head>
        <body>
            <div id="sortable-root"></div>
        </body>
        </html>
    `);
    doc.close();

    let container = doc.getElementById('sortable-root');

    for (let id of blockIds) {
        let blockData = appData.blocks_data[id];
        if (!blockData) continue;

        let wrapper = doc.createElement('div');
        wrapper.className = 'editor-block';
        if (id === selectedBlockId) wrapper.classList.add('selected');
        wrapper.dataset.id = id;
        
        // Pass block_id to render for interactive galleries
        let html = await eel.render_block_html(blockData, id)();
        wrapper.innerHTML = html;
        
        // Setup click listener inside iframe
        wrapper.onclick = (e) => {
            e.stopPropagation();
            selectBlock(id);
        };
        container.appendChild(wrapper);
    }

    // Init Sortable inside Iframe
    new Sortable(container, {
        animation: 150,
        ghostClass: 'sortable-ghost',
        onEnd: function (evt) {
            let newOrder = [];
            // Query inside the iframe document
            container.querySelectorAll('.editor-block').forEach(el => {
                newOrder.push(el.dataset.id);
            });
            appData.pages[currentPage] = newOrder;
            // Save to history and persist
            saveToHistory();
            eel.save_project_data(appData)();
        }
    });

    // Initialize interactive galleries from the parent (after DOM is ready in iframe)
    setTimeout(() => {
        let galleryContainers = doc.querySelectorAll('.interactive-gallery-container');
        galleryContainers.forEach((galleryCont) => {
            let blockId = galleryCont.getAttribute('data-block-id');
            let listEl = doc.getElementById('gallery-' + blockId);
            
            if (!listEl) return;
            
            console.log('[Parent] Initializing gallery:', blockId, 'listEl:', listEl);
            
            // Get the block data from appData
            let block = null;
            for (let bid of blockIds) {
                if (appData.blocks_data[bid] && appData.blocks_data[bid].id === blockId) {
                    block = appData.blocks_data[bid];
                    break;
                }
            }
            
            if (!block) return;
            
            // Find which images key exists (images, images_1, images_2, gallery_images, etc.)
            let imagesKey = null;
            ['images', 'images_1', 'images_2', 'images_3', 'gallery_images'].forEach(k => {
                if (!imagesKey && block[k]) imagesKey = k;
            });
            
            if (!imagesKey) return;
            
            let images = block[imagesKey];
            let selectedIdx = -1;
            
            function updateSelection() {
                let cards = listEl.querySelectorAll('.gallery-preview-card');
                cards.forEach((card, idx) => {
                    if (idx === selectedIdx) {
                        card.style.borderColor = '#1976d2';
                        card.style.opacity = '1';
                    } else {
                        card.style.borderColor = '#555';
                        card.style.opacity = '0.9';
                    }
                });
                
                // Sync with side panel image list selection
                let imagesList = document.querySelector('.gallery-images-list-sortable');
                if (imagesList) {
                    imagesList.querySelectorAll('.gallery-image-item').forEach((item, idx) => {
                        if (idx === selectedIdx) {
                            item.style.backgroundColor = '#1976d2';
                            item.style.borderLeft = '3px solid #fff';
                        } else {
                            item.style.backgroundColor = '#333';
                            item.style.borderLeft = '3px solid transparent';
                        }
                    });
                }
                
                // Update alt and caption input fields in the side panel
                if (selectedIdx >= 0 && images[selectedIdx]) {
                    let altInput = document.getElementById('alt-input-' + blockId);
                    let captionInput = document.getElementById('caption-input-' + blockId);
                    
                    let imgData = typeof images[selectedIdx] === 'string' 
                        ? {path: images[selectedIdx], alt: '', caption: ''}
                        : images[selectedIdx];
                    
                    if (altInput) altInput.value = imgData.alt || '';
                    if (captionInput) captionInput.value = imgData.caption || '';
                    
                    // Save updated images array to block
                    block.images = images;
                }
            }
            
            // Click to select
            listEl.addEventListener('click', (e) => {
                console.log('[Parent] Gallery click detected');
                let card = e.target.closest('.gallery-preview-card');
                if (!card) return;
                e.stopPropagation();
                
                // Save current selection's alt/caption BEFORE changing selection
                if (selectedIdx >= 0 && images[selectedIdx]) {
                    let altInput = document.getElementById('alt-input-' + blockId);
                    let captionInput = document.getElementById('caption-input-' + blockId);
                    if (altInput || captionInput) {
                        let item = images[selectedIdx];
                        if (typeof item === 'string') {
                            images[selectedIdx] = {
                                path: item, 
                                alt: altInput ? altInput.value : '', 
                                caption: captionInput ? captionInput.value : ''
                            };
                        } else {
                            if (altInput) item.alt = altInput.value;
                            if (captionInput) item.caption = captionInput.value;
                        }
                        console.log('[Parent] Saved previous image alt/caption before switching');
                    }
                }
                
                // Now switch to new selection
                selectedIdx = Array.from(listEl.children).indexOf(card);
                updateSelection();
            }, true);
            
            // Initialize Sortable
            try {
                console.log('[Parent] Creating Sortable on listEl');
                new Sortable(listEl, {
                    animation: 150,
                    ghostClass: 'sortable-ghost',
                    draggable: '.gallery-preview-card',
                    handle: '.drag-handle',
                    filter: '.gallery-btn-add, .gallery-btn-delete',
                    preventOnFilter: false,
                    onEnd: function(evt) {
                        console.log('[Parent] Sortable onEnd triggered');
                        // Rebuild images array from new DOM order
                        let newImgs = [];
                        listEl.querySelectorAll('.gallery-preview-card').forEach((c) => {
                            let idx = parseInt(c.getAttribute('data-idx'));
                            if (!isNaN(idx) && images[idx]) {
                                newImgs.push(images[idx]);
                            }
                        });
                        images = newImgs;
                        block[imagesKey] = images;
                        
                        // Save to appData and persist
                        saveToHistory();
                        eel.save_project_data(appData)();
                        
                        // Refresh properties panel to show updated image list
                        if (selectedBlockId === blockId) {
                            renderProperties(blockId);
                        }
                    }
                });
                console.log('[Parent] Sortable created successfully');
            } catch(err) {
                console.error('[Parent] Sortable init error:', err);
            }
            
            // Button handlers
            let addBtn = galleryCont.querySelector('.gallery-btn-add');
            let delBtn = galleryCont.querySelector('.gallery-btn-delete');
            
            console.log('[Parent] Found buttons - add:', !!addBtn, 'del:', !!delBtn);
            
            if (addBtn) {
                console.log('[Parent] Attaching add button listener');
                addBtn.addEventListener('click', async (e) => {
                    console.log('[Parent] Add button clicked!');
                    e.stopPropagation();
                    let paths = await eel.pick_images_multiselect()();
                    if (paths && paths.length > 0) {
                        paths.forEach(p => {
                            images.push({path: p.replace(/\\/g, '/'), alt: '', caption: ''});
                        });
                        block[imagesKey] = images;
                        saveToHistory();
                        eel.save_project_data(appData)();
                        
                        // Refresh properties panel to show updated image list
                        if (selectedBlockId === blockId) {
                            renderProperties(blockId);
                        }
                        
                        renderCanvas();
                    }
                }, true);
            }
            
            if (delBtn) {
                console.log('[Parent] Attaching delete button listener');
                delBtn.addEventListener('click', (e) => {
                    console.log('[Parent] Delete button clicked!');
                    e.stopPropagation();
                    if (selectedIdx >= 0) {
                        images.splice(selectedIdx, 1);
                        selectedIdx = -1;
                        block[imagesKey] = images;
                        saveToHistory();
                        eel.save_project_data(appData)();
                        
                        // Refresh properties panel to show updated image list
                        if (selectedBlockId === blockId) {
                            renderProperties(blockId);
                        }
                        
                        renderCanvas();
                    }
                }, true);
            }
        });
    }, 200);
}

function selectBlock(id) {
    selectedBlockId = id;
    
    // Update visual style inside iframe
    let doc = document.getElementById('preview-frame').contentDocument;
    doc.querySelectorAll('.editor-block').forEach(el => {
        el.classList.remove('selected');
        if(el.dataset.id === id) el.classList.add('selected');
    });

    renderProperties(id);
}

// Helper function to save gallery alt/caption before switching blocks
function saveGalleryAltCaption(blockId) {
    let block = appData.blocks_data[blockId];
    if (!block || block.type !== 'gallery') return;
    
    let altInput = document.getElementById('alt-input-' + blockId);
    let captionInput = document.getElementById('caption-input-' + blockId);
    
    // Find which image type key exists (images, images_1, etc.)
    let imageKey = 'images';
    ['images_1', 'images_2', 'images', 'gallery_images'].forEach(key => {
        if (key in block) imageKey = key;
    });
    
    let images = block[imageKey] || [];
    
    // Save the current input values to the currently selected image
    let currentIdx = -1;
    let imagesList = document.querySelector('.gallery-images-list-sortable');
    if (imagesList) {
        imagesList.querySelectorAll('.gallery-image-item').forEach((item, idx) => {
            if (item.style.backgroundColor === '#1976d2') {
                currentIdx = idx;
            }
        });
    }
    
    if (currentIdx >= 0 && images[currentIdx]) {
        let item = images[currentIdx];
        if (typeof item === 'string') {
            images[currentIdx] = {
                path: item,
                alt: altInput ? altInput.value : '',
                caption: captionInput ? captionInput.value : ''
            };
        } else {
            if (altInput) item.alt = altInput.value;
            if (captionInput) item.caption = captionInput.value;
        }
        console.log('[Gallery] Saved alt/caption before switching blocks');
    }
}


function renderProperties(id) {
    // IMPORTANT: Save any unsaved gallery alt/caption values from the PREVIOUS block
    if (selectedBlockId && selectedBlockId !== id) {
        let prevBlock = appData.blocks_data[selectedBlockId];
        if (prevBlock && prevBlock.type === 'gallery') {
            saveGalleryAltCaption(selectedBlockId);
        }
    }
    
    let panel = document.getElementById('properties-panel');
    let block = appData.blocks_data[id];
    
    panel.innerHTML = `<h3>Edit ${block.type}</h3>`;
    panel.innerHTML += `<button class="btn-danger" style="width:100%; margin-bottom:15px;" onclick="deleteBlock('${id}')">Delete Block</button>`;

    // --- HEIGHT SLIDER ---
    if (block.hasOwnProperty('height')) {
        let heightGroup = document.createElement('div');
        heightGroup.className = 'form-group';
        heightGroup.innerHTML = `<label>HEIGHT (vh) - 0=Auto</label>`;
        
        let sliderContainer = document.createElement('div');
        sliderContainer.style.display = 'flex';
        sliderContainer.style.gap = '10px';
        sliderContainer.style.alignItems = 'center';
        
        let slider = document.createElement('input');
        slider.type = 'range';
        slider.min = '0';
        slider.max = '100';
        slider.value = block.height || '0';
        slider.style.flex = '1';
        
        let valueLabel = document.createElement('span');
        valueLabel.innerText = (block.height || '0') + ' vh';
        valueLabel.style.minWidth = '45px';
        valueLabel.style.textAlign = 'right';
        
        slider.oninput = (e) => {
            valueLabel.innerText = e.target.value + ' vh';
            updateBlockData(id, 'height', parseInt(e.target.value));
        };
        
        sliderContainer.appendChild(slider);
        sliderContainer.appendChild(valueLabel);
        heightGroup.appendChild(sliderContainer);
        panel.appendChild(heightGroup);
    }

    // --- TEXT ALIGNMENT (for text-containing blocks) ---
    if (['hero','text','image_text','image_text_left'].includes(block.type)) {
        let alignGroup = document.createElement('div');
        alignGroup.className = 'form-group';
        alignGroup.innerHTML = `<label>Text Alignment</label>`;

        let sel = document.createElement('select');
        sel.style.width = '100%';
        ['left','center','right','justify'].forEach(a => {
            let opt = document.createElement('option');
            opt.value = a;
            opt.innerText = a.charAt(0).toUpperCase() + a.slice(1);
            sel.appendChild(opt);
        });
        sel.value = block.text_align || 'center';
        sel.onchange = (e) => {
            updateBlockData(id, 'text_align', e.target.value);
        };
        alignGroup.appendChild(sel);
        panel.appendChild(alignGroup);
    }

    for (let key in block) {
        if (key === 'id' || key === 'type' || key === 'height') continue;
        
        let group = document.createElement('div');
        group.className = 'form-group';
        group.innerHTML = `<label>${key.toUpperCase()}</label>`;
        
        if (key === 'image') {
            let row = document.createElement('div');
            row.style.display = 'flex';
            
            let inp = document.createElement('input'); 
            inp.value = block[key]; 
            inp.onchange = (e) => updateBlockData(id, key, e.target.value);
            
            let btn = document.createElement('button'); 
            btn.innerText = '...';
            btn.onclick = async () => {
            stopHeartbeat(); // Megállítjuk
                try {
                    let p = await eel.pick_image()();
                    if(p) { inp.value = p; updateBlockData(id, key, p); }
                } finally {
                    startHeartbeat(); // Újraindítjuk
                }
            };
            row.appendChild(inp); row.appendChild(btn);
            group.appendChild(row);
        } else if (key === 'text' || key === 'html_code') {
            let t = document.createElement('textarea');
            t.value = block[key]; t.rows = 5;
            t.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(t);
        } else if (key.startsWith('images') || key === 'gallery_images') {
             // Advanced gallery management with individual alt/caption per image
             // Handles: images, images_1, images_2, gallery_images, etc.
             let gallery = document.createElement('div');
             gallery.style.marginBottom = '15px';
             
             // Images list - DRAGGABLE
             let listLabel = document.createElement('label');
             listLabel.innerText = 'Images in Gallery (drag to reorder):';
             listLabel.style.display = 'block';
             listLabel.style.marginBottom = '5px';
             gallery.appendChild(listLabel);
             
             // Create draggable list container instead of select
             let imagesList = document.createElement('div');
             imagesList.style.width = '100%';
             imagesList.style.marginBottom = '8px';
             imagesList.style.border = '1px solid #555';
             imagesList.style.borderRadius = '4px';
             imagesList.style.minHeight = '100px';
             imagesList.style.maxHeight = '150px';
             imagesList.style.overflowY = 'auto';
             imagesList.style.backgroundColor = '#2a2a2a';
             imagesList.className = 'gallery-images-list-sortable';
             
             let selectedImageIdx = -1; // Track which image is selected
             
             let updateImagesList = () => {
                 imagesList.innerHTML = '';
                 (block[key] || []).forEach((item, idx) => {
                     let option = document.createElement('div');
                     option.draggable = true;
                     option.className = 'gallery-image-item';
                     option.dataset.idx = idx;
                     option.style.cssText = `
                         padding: 8px 10px;
                         margin: 2px 0;
                         background: ${idx === selectedImageIdx ? '#1976d2' : '#333'};
                         color: white;
                         cursor: move;
                         border-radius: 3px;
                         user-select: none;
                         border-left: 3px solid ${idx === selectedImageIdx ? '#fff' : 'transparent'};
                         transition: background 0.2s;
                     `;
                     let imgPath = typeof item === 'string' ? item : item.path;
                     option.innerText = (idx + 1) + '. ' + imgPath.split('/').pop();
                     
                     option.onclick = () => {
                         selectedImageIdx = idx;
                         updateImagesList(); // Re-render to highlight
                         
                         // Sync with preview gallery selection
                         let iframe = document.getElementById('preview-frame');
                         if (iframe && iframe.contentDocument) {
                            let previewList = iframe.contentDocument.getElementById('gallery-' + id);
                            if (previewList) {
                                let previewCards = previewList.querySelectorAll('.gallery-preview-card');
                                if (previewCards[idx]) {
                                    // Find the preview card and trigger selection
                                    let event = new MouseEvent('click', {bubbles: true, cancelable: true});
                                    previewCards[idx].dispatchEvent(event);
                                }
                            }
                         }
                         
                         let imgData = typeof item === 'string' ? {path: item, alt: '', caption: ''} : item;
                         altInput.value = imgData.alt || '';
                         captionInput.value = imgData.caption || '';
                     };
                     
                     imagesList.appendChild(option);
                 });
             };
             
             // Sortable drag/drop
             let draggedIdx = -1;
             imagesList.addEventListener('dragstart', (e) => {
                 if (e.target.classList.contains('gallery-image-item')) {
                     draggedIdx = parseInt(e.target.dataset.idx);
                 }
             });
             
             imagesList.addEventListener('dragover', (e) => {
                 e.preventDefault();
                 if (e.target.classList.contains('gallery-image-item')) {
                     e.target.style.backgroundColor = '#444';
                 }
             });
             
             imagesList.addEventListener('dragleave', (e) => {
                 if (e.target.classList.contains('gallery-image-item')) {
                     e.target.style.backgroundColor = e.target.dataset.idx == selectedImageIdx ? '#1976d2' : '#333';
                 }
             });
             
             imagesList.addEventListener('drop', (e) => {
                 e.preventDefault();
                 let dropIdx = parseInt(e.target.dataset.idx);
                 
                 if (draggedIdx !== -1 && draggedIdx !== dropIdx) {
                     // Reorder the array
                     let item = block[key][draggedIdx];
                     block[key].splice(draggedIdx, 1);
                     block[key].splice(dropIdx, 0, item);
                     selectedImageIdx = dropIdx;
                     updateImagesList();
                     saveToHistory();
                     saveProject(true);
                     renderCanvas();
                 }
                 draggedIdx = -1;
             });
             
             updateImagesList();
             gallery.appendChild(imagesList);
             
             // Add/Remove buttons
             let btnContainer = document.createElement('div');
             btnContainer.style.display = 'flex';
             btnContainer.style.gap = '5px';
             btnContainer.style.marginBottom = '10px';
             
             let addBtn = document.createElement('button');
             addBtn.innerText = '+ Image';
             addBtn.style.flex = '1';
             addBtn.onclick = async () => {
                 let paths = await eel.pick_images_multiselect()();
                 if(paths.length) {
                     paths.forEach(p => {
                         block[key].push({path:p, alt:'', caption:''});
                     });
                     updateImagesList();
                     saveToHistory();
                     saveProject(true);
                     renderCanvas();
                 }
             };
             
             let delBtn = document.createElement('button');
             delBtn.innerText = '- Delete';
             delBtn.style.flex = '1';
             delBtn.style.backgroundColor = '#d32f2f';
             delBtn.onclick = () => {
                 if (selectedImageIdx >= 0) {
                     block[key].splice(selectedImageIdx, 1);
                     selectedImageIdx = -1;
                     updateImagesList();
                     altInput.value = '';
                     captionInput.value = '';
                     saveToHistory();
                     saveProject(true);
                     renderCanvas();
                 }
             };
             
             btnContainer.appendChild(addBtn);
             btnContainer.appendChild(delBtn);
             gallery.appendChild(btnContainer);
             
             // Individual image metadata
             let metaLabel = document.createElement('label');
             metaLabel.innerText = 'Selected Image Metadata:';
             metaLabel.style.display = 'block';
             metaLabel.style.marginTop = '10px';
             metaLabel.style.marginBottom = '5px';
             metaLabel.style.fontWeight = 'bold';
             gallery.appendChild(metaLabel);
             
             let altLabel = document.createElement('label');
             altLabel.innerText = 'ALT Text (SEO):';
             altLabel.style.display = 'block';
             altLabel.style.marginBottom = '3px';
             altLabel.style.fontSize = '0.9em';
             gallery.appendChild(altLabel);
             
             let altInput = document.createElement('input');
             altInput.id = 'alt-input-' + id;
             altInput.type = 'text';
             altInput.style.width = '100%';
             altInput.style.marginBottom = '8px';
             altInput.addEventListener('input', () => {
                 if (selectedImageIdx >= 0) {
                     // Make sure we're updating the actual appData.blocks_data reference
                     let currentBlock = appData.blocks_data[id];
                     let currentKey = key;
                     let items = currentBlock[currentKey];
                     
                     if (items && items[selectedImageIdx]) {
                         let item = items[selectedImageIdx];
                         if (typeof item === 'string') {
                             items[selectedImageIdx] = {path: item, alt: altInput.value, caption: captionInput.value};
                         } else {
                             item.alt = altInput.value;
                         }
                         console.log('[Gallery] Updated alt for image', selectedImageIdx, ':', item.alt);
                     }
                     saveToHistory();
                     eel.save_project_data(appData)();
                 }
             });
             gallery.appendChild(altInput);
             
             let captionLabel = document.createElement('label');
             captionLabel.innerText = 'Caption:';
             captionLabel.style.display = 'block';
             captionLabel.style.marginBottom = '3px';
             captionLabel.style.fontSize = '0.9em';
             gallery.appendChild(captionLabel);
             
             let captionInput = document.createElement('input');
             captionInput.id = 'caption-input-' + id;
             captionInput.type = 'text';
             captionInput.style.width = '100%';
             captionInput.style.marginBottom = '8px';
             captionInput.addEventListener('input', () => {
                 if (selectedImageIdx >= 0) {
                     // Make sure we're updating the actual appData.blocks_data reference
                     let currentBlock = appData.blocks_data[id];
                     let currentKey = key;
                     let items = currentBlock[currentKey];
                     
                     if (items && items[selectedImageIdx]) {
                         let item = items[selectedImageIdx];
                         if (typeof item === 'string') {
                             items[selectedImageIdx] = {path: item, alt: altInput.value, caption: captionInput.value};
                         } else {
                             item.caption = captionInput.value;
                         }
                         console.log('[Gallery] Updated caption for image', selectedImageIdx, ':', item.caption);
                     }
                     saveToHistory();
                     eel.save_project_data(appData)();
                 }
             });
             gallery.appendChild(captionInput);
             
             // Apply to all button
             let applyAllBtn = document.createElement('button');
             applyAllBtn.innerText = 'Apply to All Images';
             applyAllBtn.style.width = '100%';
             applyAllBtn.style.marginTop = '8px';
             applyAllBtn.style.backgroundColor = '#1976d2';
             applyAllBtn.style.color = 'white';
             applyAllBtn.style.padding = '8px';
             applyAllBtn.style.border = 'none';
             applyAllBtn.style.borderRadius = '4px';
             applyAllBtn.style.cursor = 'pointer';
             applyAllBtn.onclick = () => {
                 let alt = altInput.value;
                 let caption = captionInput.value;
                 (block[key] || []).forEach((item, idx) => {
                     if (typeof item === 'string') {
                         block[key][idx] = {path: item, alt: alt, caption: caption};
                     } else {
                         item.alt = alt;
                         item.caption = caption;
                     }
                 });
                 saveToHistory();
                 saveProject(true);
                 renderCanvas();
             };
             gallery.appendChild(applyAllBtn);
             
            group.appendChild(gallery);
        } else if (key === 'grid_layout') {
            // Photo grid layout selector (row = wrap, column = single column)
            let label = document.createElement('label');
            label.innerText = 'LAYOUT';
            group.appendChild(label);
            
            let select = document.createElement('select');
            select.innerHTML = `
                <option value="row" ${block[key] === 'row' ? 'selected' : ''}>Row (Grid Wrap)</option>
                <option value="column" ${block[key] === 'column' ? 'selected' : ''}>Column (Single Column)</option>
            `;
            select.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(select);
        } else if (key === 'overlay') {
            // Overlay slider (0-100 -> stored as 0.0-1.0)
            let overlayGroup = document.createElement('div');
            overlayGroup.className = 'form-group';
            overlayGroup.innerHTML = `<label>Hero Overlay (%)</label>`;

            let oRow = document.createElement('div');
            oRow.style.display = 'flex';
            oRow.style.gap = '10px';
            oRow.style.alignItems = 'center';

            let oSlider = document.createElement('input');
            oSlider.type = 'range'; oSlider.min = 0; oSlider.max = 100;
            let current = Math.round((block[key] || 0.6) * 100);
            oSlider.value = current;
            oSlider.style.flex = '1';

            let oLabel = document.createElement('span');
            oLabel.innerText = current + ' %';
            oLabel.style.minWidth = '50px'; oLabel.style.textAlign = 'right';

            oSlider.oninput = (e) => {
                let pct = parseInt(e.target.value);
                oLabel.innerText = pct + ' %';
                updateBlockData(id, key, (pct/100));
            };

            oRow.appendChild(oSlider); oRow.appendChild(oLabel);
            overlayGroup.appendChild(oRow);
            group.appendChild(overlayGroup);
        } else if (key === 'button_text' || key === 'button_url' || key === 'button_color' || key === 'button_align') {
            // Button block properties
            let label = document.createElement('label');
            let labelText = key.replace('button_', '').replace(/_/g, ' ').toUpperCase();
            label.innerText = labelText;
            group.appendChild(label);
            
            let i = document.createElement('input');
            i.value = block[key] || '';
            if (key === 'button_color') i.type = 'color';
            i.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(i);
        } else if (key === 'container_layout') {
            // Container layout dropdown
            let label = document.createElement('label');
            label.innerText = 'LAYOUT';
            group.appendChild(label);
            
            let select = document.createElement('select');
            select.innerHTML = `
                <option value="row" ${block[key] === 'row' ? 'selected' : ''}>Row (Horizontal)</option>
                <option value="column" ${block[key] === 'column' ? 'selected' : ''}>Column (Vertical)</option>
            `;
            select.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(select);
        } else if (key === 'container_gap') {
            // Container gap
            let label = document.createElement('label');
            label.innerText = 'GAP (spacing)';
            group.appendChild(label);
            
            let i = document.createElement('input');
            i.value = block[key] || '20px';
            i.placeholder = '20px';
            i.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(i);
        } else if (key === 'container_items') {
            // Container items manager - special UI
            let label = document.createElement('label');
            label.innerText = 'ITEMS (max 5)';
            label.style.display = 'block';
            label.style.fontWeight = 'bold';
            label.style.marginBottom = '10px';
            group.appendChild(label);
            
            let itemsList = document.createElement('div');
            itemsList.style.border = '1px solid #555';
            itemsList.style.borderRadius = '4px';
            itemsList.style.padding = '10px';
            itemsList.style.marginBottom = '10px';
            itemsList.style.maxHeight = '200px';
            itemsList.style.overflowY = 'auto';
            itemsList.style.backgroundColor = '#2a2a2a';
            
            let items = block[key] || [];
            
            if (items.length === 0) {
                let emptyMsg = document.createElement('p');
                emptyMsg.innerText = 'No items yet. Add one below.';
                emptyMsg.style.color = '#888';
                itemsList.appendChild(emptyMsg);
            } else {
                items.forEach((item, idx) => {
                    let itemDiv = document.createElement('div');
                    itemDiv.style.padding = '8px';
                    itemDiv.style.marginBottom = '5px';
                    itemDiv.style.background = '#333';
                    itemDiv.style.borderRadius = '3px';
                    itemDiv.style.display = 'flex';
                    itemDiv.style.justifyContent = 'space-between';
                    itemDiv.style.alignItems = 'center';
                    
                    let itemText = document.createElement('span');
                    itemText.innerText = `${idx + 1}. [${item.type || 'text'}] ${(item.content || item.text || item.url || '').substring(0, 30)}...`;
                    itemText.style.flex = '1';
                    itemDiv.appendChild(itemText);
                    
                    let editBtn = document.createElement('button');
                    editBtn.innerText = '✎';
                    editBtn.style.padding = '3px 8px';
                    editBtn.style.marginRight = '5px';
                    editBtn.style.cursor = 'pointer';
                    editBtn.onclick = () => editContainerItem(id, idx, block);
                    itemDiv.appendChild(editBtn);
                    
                    let delBtn = document.createElement('button');
                    delBtn.innerText = '✕';
                    delBtn.style.padding = '3px 8px';
                    delBtn.style.backgroundColor = '#d32f2f';
                    delBtn.style.cursor = 'pointer';
                    delBtn.onclick = () => {
                        items.splice(idx, 1);
                        updateBlockData(id, 'container_items', items);
                    };
                    itemDiv.appendChild(delBtn);
                    
                    itemsList.appendChild(itemDiv);
                });
            }
            
            group.appendChild(itemsList);
            
            // Add item button
            let addItemBtn = document.createElement('button');
            addItemBtn.innerText = '+ Add Item';
            addItemBtn.style.width = '100%';
            addItemBtn.style.padding = '10px';
            addItemBtn.style.backgroundColor = '#007acc';
            addItemBtn.style.color = 'white';
            addItemBtn.style.border = 'none';
            addItemBtn.style.borderRadius = '4px';
            addItemBtn.style.cursor = 'pointer';
            addItemBtn.onclick = () => addContainerItem(id, block);
            group.appendChild(addItemBtn);
        } else if (key === 'container_height') {
            // Container height slider (for multiple rows)
            let label = document.createElement('label');
            label.innerText = 'CONTAINER HEIGHT (vh)';
            group.appendChild(label);
            
            let heightSlider = document.createElement('input');
            heightSlider.type = 'range';
            heightSlider.min = '0';
            heightSlider.max = '100';
            heightSlider.value = block[key] || '0';
            heightSlider.style.width = '100%';
            heightSlider.style.marginBottom = '5px';
            
            let heightLabel = document.createElement('div');
            heightLabel.style.fontSize = '0.9em';
            heightLabel.style.color = '#aaa';
            heightLabel.style.marginBottom = '10px';
            heightLabel.innerText = (block[key] || '0') + ' vh (0 = auto)';
            
            heightSlider.onchange = (e) => {
                let val = e.target.value;
                heightLabel.innerText = val + ' vh (0 = auto)';
                updateBlockData(id, key, parseInt(val));
            };
            
            heightSlider.oninput = (e) => {
                heightLabel.innerText = e.target.value + ' vh (0 = auto)';
            };
            
            group.appendChild(heightSlider);
            group.appendChild(heightLabel);
        } else if ((key === 'alt' || key === 'description') && ['image_text', 'image_text_left'].includes(block.type)) {
            // Special handling for alt and description in image_text blocks
            let label = document.createElement('label');
            let labelText = key === 'alt' ? 'ALT Text (Image Description for SEO)' : 'Image Caption/Description';
            label.innerText = labelText;
            group.appendChild(label);
            
            let textarea = document.createElement('textarea');
            textarea.value = block[key] || '';
            textarea.rows = key === 'description' ? 4 : 2;
            textarea.style.width = '100%';
            textarea.style.padding = '5px';
            textarea.style.borderRadius = '3px';
            // Use onblur (when user leaves the field) instead of onchange to avoid constant re-renders
            textarea.onblur = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(textarea);
        } else {
            let i = document.createElement('input');
            i.value = block[key];
            i.onchange = (e) => updateBlockData(id, key, e.target.value);
            group.appendChild(i);
        }
        panel.appendChild(group);
    }
}

window.updateBlockData = function(id, key, val) {
    appData.blocks_data[id][key] = val;
    saveToHistory();
    saveProject(true);
    renderCanvas();
};

window.addImages = async function(id) {
    stopHeartbeat(); // STOP
    try {
        let paths = await eel.pick_images_multiselect()();
        if(paths.length) {
            paths.forEach(p => appData.blocks_data[id].images.push({path:p, alt:'', caption:''}));
            saveToHistory();
            saveProject(true);
            renderCanvas();
        }
    } finally {
        startHeartbeat(); // START
    }
};

window.addBlock = async function(type) {
    let res = await eel.create_block(type, appData.theme)();
    appData.blocks_data[res.id] = res.content;
    let page = document.getElementById('page-select').value;
    appData.pages[page].push(res.id);
    saveToHistory();
    saveProject(true);
    renderCanvas();
};

window.deleteBlock = function(id) {
    if(!confirm("Delete?")) return;
    let page = document.getElementById('page-select').value;
    appData.pages[page] = appData.pages[page].filter(x => x !== id);
    delete appData.blocks_data[id];
    document.getElementById('properties-panel').innerHTML = '';
    saveToHistory();
    saveProject(true);
    renderCanvas();
};

// Container item management
window.addContainerItem = function(blockId, block) {
    if (!block.container_items) block.container_items = [];
    if (block.container_items.length >= 5) {
        alert('Maximum 5 items allowed');
        return;
    }
    
    showContainerItemEditor(blockId, block, null);
};

window.editContainerItem = function(blockId, itemIdx, block) {
    let item = block.container_items[itemIdx];
    if (!item) return;
    
    showContainerItemEditor(blockId, block, itemIdx);
};

window.showContainerItemEditor = function(blockId, block, itemIdx) {
    const modal = document.createElement('div');
    modal.id = 'container-item-editor';
    modal.style.cssText = `
        position: fixed;
        top: 0; left: 0;
        width: 100%; height: 100%;
        background: rgba(0,0,0,0.7);
        display: flex;
        align-items: center;
        justify-content: center;
        z-index: 9999;
    `;
    
    const dialog = document.createElement('div');
    dialog.style.cssText = `
        background: #333;
        border-radius: 8px;
        padding: 20px;
        width: 90%;
        max-width: 600px;
        max-height: 85vh;
        overflow-y: auto;
        box-shadow: 0 4px 20px rgba(0,0,0,0.5);
    `;
    
    const isEdit = itemIdx !== null;
    let item = isEdit ? block.container_items[itemIdx] : { type: 'text', content: '', margin: '10px', align_horizontal: 'center', align_vertical: 'center' };
    
    // Ensure alignment properties exist (backward compatibility for old items)
    if (!item.align_horizontal) item.align_horizontal = 'center';
    if (!item.align_vertical) item.align_vertical = 'center';
    
    dialog.innerHTML = `
        <h3 style="margin-top: 0; color: #fff;">${isEdit ? 'Edit' : 'Add'} Item</h3>
        
        <div style="margin-bottom: 15px;">
            <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Item Type</label>
            <select id="item-type" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555;">
                <option value="text" ${item.type === 'text' ? 'selected' : ''}>Text</option>
                <option value="image" ${item.type === 'image' ? 'selected' : ''}>Image</option>
                <option value="button" ${item.type === 'button' ? 'selected' : ''}>Button</option>
                <option value="html" ${item.type === 'html' ? 'selected' : ''}>Custom HTML</option>
            </select>
        </div>
        
        <div id="item-properties-container" style="border-top: 1px solid #555; padding-top: 15px;"></div>
        
        <div style="margin-top: 15px; border-top: 1px solid #555; padding-top: 15px;">
            <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Alignment</label>
            
            <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 10px; margin-bottom: 10px;">
                <div>
                    <label style="display: block; margin-bottom: 3px; color: #888; font-size: 0.9em;">Horizontal</label>
                    <select id="item-align-horizontal" style="width: 100%; padding: 6px; background: #222; color: white; border: 1px solid #555; font-size: 0.9em;">
                        <option value="flex-start" ${item.align_horizontal === 'flex-start' ? 'selected' : ''}>← Left</option>
                        <option value="center" ${item.align_horizontal === 'center' ? 'selected' : ''}>↔ Center</option>
                        <option value="flex-end" ${item.align_horizontal === 'flex-end' ? 'selected' : ''}>→ Right</option>
                    </select>
                </div>
                <div>
                    <label style="display: block; margin-bottom: 3px; color: #888; font-size: 0.9em;">Vertical</label>
                    <select id="item-align-vertical" style="width: 100%; padding: 6px; background: #222; color: white; border: 1px solid #555; font-size: 0.9em;">
                        <option value="flex-start" ${item.align_vertical === 'flex-start' ? 'selected' : ''}>↑ Top</option>
                        <option value="center" ${item.align_vertical === 'center' ? 'selected' : ''}>↕ Center</option>
                        <option value="flex-end" ${item.align_vertical === 'flex-end' ? 'selected' : ''}>↓ Bottom</option>
                    </select>
                </div>
            </div>
        </div>
        
        <div style="margin-top: 15px;">
            <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Margin (px)</label>
            <input id="item-margin" type="text" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.margin || '10px'}" placeholder="10px">
            <small style="color: #888;">Space around the item (e.g., 10px, 5px 10px)</small>
        </div>
        
        <div style="display: flex; gap: 10px; margin-top: 20px;">
            <button id="save-item" style="flex: 1; padding: 10px; background: #007acc; color: white; border: none; border-radius: 4px; cursor: pointer;">Save</button>
            <button id="cancel-item" style="flex: 1; padding: 10px; background: #555; color: white; border: none; border-radius: 4px; cursor: pointer;">Cancel</button>
        </div>
    `;
    
    modal.appendChild(dialog);
    document.body.appendChild(modal);
    
    const renderItemProperties = (type) => {
        const container = document.getElementById('item-properties-container');
        container.innerHTML = '';
        
        if (type === 'text') {
            container.innerHTML = `
                <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Text Content</label>
                <textarea id="item-content" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; min-height: 100px; font-family: monospace; box-sizing: border-box;">${item.content || ''}</textarea>
                <small style="color: #888;">Supports HTML tags</small>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Text Align</label>
                    <select id="item-text-align" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555;">
                        <option value="left" ${item.text_align === 'left' ? 'selected' : ''}>Left</option>
                        <option value="center" ${item.text_align === 'center' ? 'selected' : ''}>Center</option>
                        <option value="right" ${item.text_align === 'right' ? 'selected' : ''}>Right</option>
                    </select>
                </div>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Font Size (px)</label>
                    <input id="item-font-size" type="number" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.font_size || '16'}" min="10" max="48">
                </div>
            `;
        } else if (type === 'image') {
            container.innerHTML = `
                <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Image Path</label>
                <input id="item-src" type="text" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.src || ''}" placeholder="kepek/image.jpg">
                <small style="color: #888;">Path relative to project root (e.g., kepek/photo.jpg)</small>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Alt Text</label>
                    <input id="item-alt" type="text" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.alt || ''}" placeholder="Image description">
                </div>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Border Radius (px)</label>
                    <input id="item-border-radius" type="number" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.border_radius || '5'}" min="0" max="50">
                </div>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Max Height (px)</label>
                    <div style="display: flex; gap: 10px; align-items: center;">
                        <input id="item-max-height" type="range" style="flex: 1; cursor: pointer;" min="50" max="600" value="${item.max_height || '300'}" step="10">
                        <span id="max-height-label" style="min-width: 60px; text-align: right; color: #aaa;">${item.max_height || '300'}px</span>
                    </div>
                    <small style="color: #888;">Width adapts to container, height limited to this value</small>
                </div>
            `;
            
            // Update label when slider changes
            setTimeout(() => {
                const slider = document.getElementById('item-max-height');
                const label = document.getElementById('max-height-label');
                if (slider) {
                    slider.oninput = (e) => {
                        label.innerText = e.target.value + 'px';
                    };
                }
            }, 10);

        } else if (type === 'button') {
            container.innerHTML = `
                <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Button Text</label>
                <input id="item-text" type="text" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.text || 'Click Me'}" placeholder="Button text">
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Button URL</label>
                    <input id="item-url" type="text" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; box-sizing: border-box;" value="${item.url || '#'}" placeholder="https://example.com">
                </div>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Button Color</label>
                    <input id="item-color" type="color" style="width: 100%; padding: 8px; background: #222; border: 1px solid #555; height: 40px; cursor: pointer;" value="${item.color || '#007acc'}">
                </div>
                
                <div style="margin-top: 15px;">
                    <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Button Align</label>
                    <select id="item-button-align" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555;">
                        <option value="left" ${item.button_align === 'left' ? 'selected' : ''}>Left</option>
                        <option value="center" ${item.button_align === 'center' ? 'selected' : ''}>Center</option>
                        <option value="right" ${item.button_align === 'right' ? 'selected' : ''}>Right</option>
                    </select>
                </div>
            `;
        } else if (type === 'html') {
            container.innerHTML = `
                <label style="display: block; margin-bottom: 5px; color: #aaa; font-weight: bold;">Custom HTML</label>
                <textarea id="item-content" style="width: 100%; padding: 8px; background: #222; color: white; border: 1px solid #555; min-height: 100px; font-family: monospace; box-sizing: border-box;">${item.content || ''}</textarea>
                <small style="color: #888;">Enter any valid HTML code</small>
            `;
        }
    };
    
    renderItemProperties(item.type);
    
    document.getElementById('item-type').onchange = (e) => {
        renderItemProperties(e.target.value);
    };
    
    document.getElementById('save-item').onclick = () => {
        const type = document.getElementById('item-type').value;
        const newItem = { 
            type, 
            margin: document.getElementById('item-margin').value || '10px',
            align_horizontal: document.getElementById('item-align-horizontal').value || 'center',
            align_vertical: document.getElementById('item-align-vertical').value || 'center'
        };
        
        if (type === 'text') {
            newItem.content = document.getElementById('item-content').value;
            newItem.text_align = document.getElementById('item-text-align').value;
            newItem.font_size = document.getElementById('item-font-size').value;
        } else if (type === 'image') {
            newItem.src = document.getElementById('item-src').value;
            newItem.alt = document.getElementById('item-alt').value;
            newItem.border_radius = document.getElementById('item-border-radius').value;
            newItem.max_height = parseInt(document.getElementById('item-max-height').value) || 300;
        } else if (type === 'button') {
            newItem.text = document.getElementById('item-text').value;
            newItem.url = document.getElementById('item-url').value;
            newItem.color = document.getElementById('item-color').value;
            newItem.button_align = document.getElementById('item-button-align').value;
        } else if (type === 'html') {
            newItem.content = document.getElementById('item-content').value;
        }
        
        if (isEdit) {
            block.container_items[itemIdx] = newItem;
        } else {
            block.container_items.push(newItem);
        }
        
        modal.remove();
        updateBlockData(blockId, 'container_items', block.container_items);
    };
    
    document.getElementById('cancel-item').onclick = () => {
        modal.remove();
    };
    
    modal.onclick = (e) => {
        if (e.target === modal) modal.remove();
    };
};

window.saveProject = async function(silent) {
    await eel.save_project_data(appData)();
    if(!silent) console.log("Saved");
};

// UI helpers: transient modal + toast for generation status
window.showGeneratingModal = function(message) {
    // remove existing if any
    let existing = document.getElementById('generating-modal');
    if (existing) existing.remove();

    const modal = document.createElement('div');
    modal.id = 'generating-modal';
    modal.style.position = 'fixed';
    modal.style.inset = '0';
    modal.style.display = 'flex';
    modal.style.alignItems = 'center';
    modal.style.justifyContent = 'center';
    modal.style.background = 'rgba(0,0,0,0.45)';
    modal.style.zIndex = 99999;

    const box = document.createElement('div');
    box.style.background = '#fff';
    box.style.color = '#111';
    box.style.padding = '18px 22px';
    box.style.borderRadius = '8px';
    box.style.boxShadow = '0 4px 18px rgba(0,0,0,0.3)';
    box.style.maxWidth = '90%';
    box.style.display = 'flex';
    box.style.alignItems = 'center';
    box.style.gap = '12px';

    const spinner = document.createElement('div');
    spinner.style.width = '18px';
    spinner.style.height = '18px';
    spinner.style.border = '3px solid #ccc';
    spinner.style.borderTop = '3px solid #333';
    spinner.style.borderRadius = '50%';
    spinner.style.animation = 'genspin 1s linear infinite';

    const msg = document.createElement('div');
    msg.style.fontSize = '14px';
    msg.style.whiteSpace = 'nowrap';
    msg.innerText = message || 'Generating...';

    box.appendChild(spinner);
    box.appendChild(msg);
    modal.appendChild(box);

    // simple animation keyframes (insert once)
    if (!document.getElementById('genspin-style')) {
        const style = document.createElement('style');
        style.id = 'genspin-style';
        style.innerHTML = `@keyframes genspin { from { transform: rotate(0deg) } to { transform: rotate(360deg) } }`;
        document.head.appendChild(style);
    }

    document.body.appendChild(modal);
    return modal;
};

window.hideGeneratingModal = function() {
    let existing = document.getElementById('generating-modal');
    if (existing) existing.remove();
};

window.showToast = function(message, timeout) {
    timeout = typeof timeout === 'number' ? timeout : 3500;
    const id = 'gen-toast';
    let toast = document.getElementById(id);
    if (toast) toast.remove();
    toast = document.createElement('div');
    toast.id = id;
    toast.style.position = 'fixed';
    toast.style.right = '18px';
    toast.style.bottom = '18px';
    toast.style.background = 'rgba(0,0,0,0.85)';
    toast.style.color = '#fff';
    toast.style.padding = '10px 14px';
    toast.style.borderRadius = '6px';
    toast.style.boxShadow = '0 6px 18px rgba(0,0,0,0.3)';
    toast.style.zIndex = 100000;
    toast.style.maxWidth = '80%';
    toast.style.fontSize = '13px';
    toast.style.opacity = '0';
    toast.style.transition = 'opacity 240ms ease';
    toast.innerText = message || '';
    document.body.appendChild(toast);
    // fade in
    requestAnimationFrame(() => { toast.style.opacity = '1'; });
    setTimeout(() => {
        toast.style.opacity = '0';
        setTimeout(() => { try { toast.remove(); } catch(e){} }, 300);
    }, timeout);
};

window.generateSite = async function() {
    stopHeartbeat();
    // Show transient modal while generation runs
    try {
        showGeneratingModal('Generating site and images — this may take a while...');
        const result = await eel.generate_website()();
        hideGeneratingModal();
        // show short toast with result text
        showToast(result || 'Generation finished', 6000);
    } catch (err) {
        hideGeneratingModal();
        // fallback to alert for errors
        alert('Generation failed: ' + (err && err.message ? err.message : err));
    } finally {
        // 2. VISSZAKAPCSOLJUK, ha végzett
        startHeartbeat(); 
    }
};

function updatePageSelector() {
    let sel = document.getElementById('page-select');
    sel.innerHTML = '';
    // Use page_order if available, otherwise iterate over pages
    let pageOrder = appData.page_order || Object.keys(appData.pages);
    for(let k of pageOrder) {
        if (!appData.pages[k]) continue; // Skip if page no longer exists
        let opt = document.createElement('option');
        let hidden = (appData.page_hidden && appData.page_hidden[k]);
        opt.value = k;
        opt.innerText = (appData.page_titles[k] || k) + (hidden ? ' (hidden)' : '');
        sel.appendChild(opt);
    }
}
window.changePage = (v) => { eel.set_current_page(v)(); renderCanvas(); };

// Page management helpers: rename, duplicate, hide/unhide
window.renamePage = async function() {
    let sel = document.getElementById('page-select');
    let oldKey = sel.value;
    if(!oldKey) return;
    let newName = prompt('New page name (display name):', appData.page_titles[oldKey] || oldKey);
    if(!newName) return;
    // generate slug for key - removes accents properly
    let slug = slugifyPageName(newName);
    // If renaming the special 'index' page, do NOT change its key (filename).
    if (oldKey === 'index') {
        appData.page_titles[oldKey] = newName;
        await eel.save_project_data(appData)();
        updatePageSelector();
        renderProperties(oldKey);
        renderCanvas();
        return;
    }
    if(slug === oldKey) {
        // only update title
        appData.page_titles[oldKey] = newName;
        await eel.save_project_data(appData)();
        updatePageSelector();
        renderProperties(oldKey);
        renderCanvas();
        return;
    }
    
    // Ensure slug is unique - if conflict, append number
    let finalSlug = slug;
    let counter = 1;
    while(appData.pages[finalSlug]) {
        finalSlug = slug + '-' + counter;
        counter++;
    }
    slug = finalSlug;
    
    // move page entry
    appData.pages[slug] = appData.pages[oldKey];
    delete appData.pages[oldKey];
    // move title
    appData.page_titles[slug] = newName;
    delete appData.page_titles[oldKey];
    // move hidden state
    if(!appData.page_hidden) appData.page_hidden = {};
    appData.page_hidden[slug] = appData.page_hidden[oldKey] || false;
    delete appData.page_hidden[oldKey];
    // update page_order to reflect key change
    if(!appData.page_order) appData.page_order = Object.keys(appData.pages);
    let idx = appData.page_order.indexOf(oldKey);
    if(idx >= 0) {
        appData.page_order[idx] = slug;
    }
    // update current page selection
    await eel.save_project_data(appData)();
    await eel.update_page_order(appData.page_order)();
    updatePageSelector();
    document.getElementById('page-select').value = slug;
    renderCanvas();
};

// Helper function to slugify text - removes accents and special characters
function slugifyPageName(text) {
    // Normalize to NFD (decomposed form), keeping base characters
    let normalized = text.normalize('NFD');
    // Remove combining marks (accents)
    let cleaned = normalized.replace(/[\u0300-\u036f]/g, '');
    // Lowercase and remove special characters (keep only alphanumeric, spaces, hyphens)
    let slug = cleaned.toLowerCase().replace(/[^a-z0-9\s-]/g, '').trim();
    // Replace spaces and multiple hyphens with single hyphen
    return slug.replace(/[\s-]+/g, '-');
}

window.editPageDescription = function() {
    let sel = document.getElementById('page-select');
    let pageKey = sel.value;
    if (!pageKey) return;
    
    // Ensure page_descriptions object exists
    if (!appData.page_descriptions) appData.page_descriptions = {};
    
    let currentDesc = appData.page_descriptions[pageKey] || "";
    let newDesc = prompt(`Edit Meta Description for "${appData.page_titles[pageKey] || pageKey}":\n\n(Leave empty to use global description)`, currentDesc);
    
    if (newDesc !== null) {
        appData.page_descriptions[pageKey] = newDesc;
        saveToHistory();
        saveProject(true);
    }
};

window.duplicatePage = async function() {
    let sel = document.getElementById('page-select');
    let key = sel.value;
    if(!key) return;
    let newName = prompt('Duplicate page name (display name):', (appData.page_titles[key] || key) + ' Copy');
    if(!newName) return;
    // generate slug for key - removes accents properly
    let slug = slugifyPageName(newName);
    
    // Ensure slug is unique - if conflict, append number
    let finalSlug = slug;
    let counter = 1;
    while(appData.pages[finalSlug]) {
        finalSlug = slug + '-' + counter;
        counter++;
    }
    slug = finalSlug;
    // deep copy blocks: create new block ids and copy block data
    let oldBlocks = appData.pages[key] || [];
    let newBlockIds = [];
    for(let bid of oldBlocks) {
        let block = appData.blocks_data[bid];
        if(!block) continue;
        // generate new id
        let newId = block.type + '_' + (Date.now().toString(36)) + '_' + Math.floor(Math.random()*10000).toString(36);
        // shallow copy of data
        let copy = JSON.parse(JSON.stringify(block));
        copy.id = newId;
        appData.blocks_data[newId] = copy;
        newBlockIds.push(newId);
    }
    appData.pages[slug] = newBlockIds;
    if(!appData.page_titles) appData.page_titles = {};
    appData.page_titles[slug] = newName;
    if(!appData.page_hidden) appData.page_hidden = {};
    appData.page_hidden[slug] = false;
    // Copy page description or initialize empty
    if(!appData.page_descriptions) appData.page_descriptions = {};
    appData.page_descriptions[slug] = appData.page_descriptions[key] || "";
    // Add new page to page_order
    if(!appData.page_order) appData.page_order = Object.keys(appData.pages);
    if(!appData.page_order.includes(slug)) {
        appData.page_order.push(slug);
    }
    await eel.save_project_data(appData)();
    await eel.update_page_order(appData.page_order)();
    updatePageSelector();
    document.getElementById('page-select').value = slug;
    renderCanvas();
};

window.toggleHidePage = async function() {
    let sel = document.getElementById('page-select');
    let key = sel.value;
    if(!key) return;
    if(!appData.page_hidden) appData.page_hidden = {};
    appData.page_hidden[key] = !appData.page_hidden[key];
    await eel.save_project_data(appData)();
    // Reload appData from server to ensure consistency
    appData = await eel.get_project_data()();
    updatePageSelector();
    document.getElementById('page-select').value = key;
    renderCanvas();
};

window.movePageUp = async function() {
    let sel = document.getElementById('page-select');
    let key = sel.value;
    if(!key) return;
    if(!appData.page_order) appData.page_order = Object.keys(appData.pages);
    let idx = appData.page_order.indexOf(key);
    if(idx > 0) {
        // Swap with previous
        let temp = appData.page_order[idx - 1];
        appData.page_order[idx - 1] = appData.page_order[idx];
        appData.page_order[idx] = temp;
        await eel.update_page_order(appData.page_order)();
        // Reload appData to ensure consistency
        appData = await eel.get_project_data()();
        updatePageSelector();
        document.getElementById('page-select').value = key;
        renderCanvas();
    }
};

window.movePageDown = async function() {
    let sel = document.getElementById('page-select');
    let key = sel.value;
    if(!key) return;
    if(!appData.page_order) appData.page_order = Object.keys(appData.pages);
    let idx = appData.page_order.indexOf(key);
    if(idx < appData.page_order.length - 1) {
        // Swap with next
        let temp = appData.page_order[idx + 1];
        appData.page_order[idx + 1] = appData.page_order[idx];
        appData.page_order[idx] = temp;
        await eel.update_page_order(appData.page_order)();
        // Reload appData to ensure consistency
        appData = await eel.get_project_data()();
        updatePageSelector();
        document.getElementById('page-select').value = key;
        renderCanvas();
    }
};
window.addPage = async () => {
    let n = prompt("Name?");
    if(n) {
        let r = await eel.add_new_page(n)();
        if(r.success) { 
            appData = await eel.get_project_data()(); 
            // Ensure new page is added to page_order
            if(!appData.page_order) appData.page_order = Object.keys(appData.pages);
            if(!appData.page_order.includes(r.key)) {
                appData.page_order.push(r.key);
                await eel.update_page_order(appData.page_order)();
            }
            // Ensure page_descriptions is initialized for new page
            if(!appData.page_descriptions) appData.page_descriptions = {};
            if(!appData.page_descriptions[r.key]) appData.page_descriptions[r.key] = "";
            updatePageSelector(); 
            document.getElementById('page-select').value = r.key; 
            saveToHistory(); 
            renderCanvas(); 
        }
    }
};
window.deletePage = async () => {
    let p = document.getElementById('page-select').value;
    if(confirm("Delete "+p+"?")) {
        let r = await eel.delete_page(p)();
        if(r.success) { saveToHistory(); loadProject(); }
    }
};
window.showFaq = function() {
    let iframe = document.getElementById('preview-frame');
    // Egyszerűen átállítjuk az iframe forrását a helyi fájlra
    iframe.src = "faq.html";
};


